import { CFG } from './config.js';
import { strat } from './strategy.js';
import { saveState } from './storage.js';

const bips = (p) => Math.round(p * 10000);
function notional(qty, price){ return qty * price; }
function qtyFromNotional(notional, price, fee){ return (notional / price) * (1 - fee); }
function weightedAvg(lots){ const q = lots.reduce((a,b)=>a+b.qty,0); const v = lots.reduce((a,b)=>a+b.qty*b.entry,0); return v / q; }

export const engine = {
  onEntry(state, payload){
    const { price, atr, atrPct, barIndex } = payload;
    if (state.pos) return { info: 'already in position' };
    if (state.cooldownUntilBar !== -1 && barIndex <= state.cooldownUntilBar) return { info: 'in cooldown' };

    const f0 = strat.sizingFromATR(atrPct);
    const deploy = state.equityUSDT * f0;
    if (deploy < CFG.MIN_NOTIONAL) return { warn: 'deploy below MIN_NOTIONAL' };

    const qty = qtyFromNotional(deploy, price, CFG.FEE);
    const stop0 = strat.makeInitialStop(price, atr);
    const rbase = price - stop0;
    const riskBudget = qty * (price - stop0);

    state.pos = {
      lots: [{ qty, entry: price }],
      stop: stop0,
      rbase,
      riskBudget,
      trailing: false,
      trailDeltaBips: null,
      lastHigh: price,
      entryBar: barIndex,
      f0
    };

    state.equityUSDT -= notional(qty, price) * (1 + CFG.FEE);
    state.logs.push({ t: Date.now(), type: 'BUY', info: `ENTRY @${price.toFixed(2)} qty=${qty}`, f0 });
    saveState(state);
    return { ok: true, action: 'BUY', price, qty };
  },

  onTick(state, payload){
    if (!state.pos) return { info: 'flat' };
    const { price, high, low, atrPct, adx14, barIndex } = payload;

    const P = state.pos;
    const Q = P.lots.reduce((a,b)=>a+b.qty,0);
    const avgEntry = weightedAvg(P.lots);

    if (high && (!P.lastHigh || high > P.lastHigh)) P.lastHigh = high;

    if (!P.trailing && low <= P.stop) {
      const exit = P.stop;
      state.equityUSDT += Q * exit * (1 - CFG.FEE);
      state.logs.push({ t: Date.now(), type: 'SL', info: `EXIT SL @${exit.toFixed(2)}` });
      state.pos = null;
      state.cooldownUntilBar = (barIndex ?? 0) + CFG.COOLDOWN_BARS;
      saveState(state);
      return { exit: 'SL', price: exit };
    }

    if (P.trailing && low <= P.stop) {
      const exit = P.stop;
      state.equityUSDT += Q * exit * (1 - CFG.FEE);
      state.logs.push({ t: Date.now(), type: 'TRAIL_EXIT', info: `EXIT TRAIL @${exit.toFixed(2)}` });
      state.pos = null;
      state.cooldownUntilBar = (barIndex ?? 0) + CFG.COOLDOWN_BARS;
      saveState(state);
      return { exit: 'TRAIL', price: exit };
    }

    // -------- Adds (pre-trail) using frozen R_base --------
    if (!P.trailing) {
      const R = P.rbase;
      const dist = (P.lastHigh || price) - avgEntry;

      let addThresholds = [], targets = [];
      if (P.f0 <= 0.20) { addThresholds = [0.6, 1.0, 1.3]; targets = [0.40, 0.80, 1.00]; }
      else if (P.f0 <= 0.40) { addThresholds = [0.9, 1.2]; targets = [0.80, 1.00]; }
      else if (P.f0 <= 0.80) { addThresholds = [1.0]; targets = [1.00]; }
      else { addThresholds = []; targets = []; }

      const executedAdds = P.lots.length - 1;
      const nextIdx = Math.min(executedAdds, addThresholds.length);

      if (nextIdx < addThresholds.length) {
        const thr = addThresholds[nextIdx];
        const targetFrac = targets[nextIdx];
        if (dist < (CFG.FLIP_R - CFG.NEAR_FLIP_GAP) * R) {
          if ((P.lastHigh || price) >= (avgEntry + thr * R) && payload.filtersOk) {
            const equityNow = state.equityUSDT + Q * price;
            const currentNotional = Q * price;
            let addNotional = Math.max(0, targetFrac * equityNow - currentNotional);
            addNotional = Math.min(addNotional, state.equityUSDT);
            if (addNotional >= CFG.MIN_NOTIONAL) {
              const addQty = (addNotional / price) * (1 - CFG.FEE);
              const newQ = Q + addQty;
              const weightedEntries = P.lots.reduce((a,b)=>a+b.qty*b.entry,0) + addQty*price;
              const S_req = (weightedEntries - P.riskBudget) / newQ;
              if (S_req < price * (1 - 0.001) && S_req >= P.stop) {
                state.equityUSDT -= addNotional * (1 + CFG.FEE);
                P.lots.push({ qty: addQty, entry: price });
                P.stop = S_req;
                state.logs.push({ t: Date.now(), type: 'BUY', info: `ADD@${thr}R -> ${(targetFrac*100).toFixed(0)}% @${price.toFixed(2)}; new SL ${S_req.toFixed(2)}` });
              }
            }
          }
        }
      }

      // -------- Flip to trailing @ +2.0R (frozen) --------
      if ((P.lastHigh || price) >= (avgEntry + CFG.FLIP_R * R)) {
        const tpct = strat.trailPct(atrPct, adx14);
        P.trailing = true;
        P.trailDeltaBips = Math.round(tpct * 10000);
        const trailPrice = price * (1 - tpct);
        P.stop = Math.max(P.stop, trailPrice);
        state.logs.push({ t: Date.now(), type: 'TRAIL_ON', info: `trail ${(tpct*100).toFixed(2)}% start; stop ${P.stop.toFixed(2)}` });
      }
    } else {
      const tpct = strat.trailPct(atrPct, adx14);
      const stageMult = ((P.lastHigh || price) >= (weightedAvg(P.lots) + CFG.FLIP_R * P.rbase)) ? 0.75 : 1.0;
      const effPct = tpct * stageMult;
      const candidate = (P.lastHigh || price) * (1 - effPct);
      if (candidate > P.stop) {
        P.stop = candidate;
        P.trailDeltaBips = Math.round(effPct * 10000);
        state.logs.push({ t: Date.now(), type: 'TRAIL_RAISE', info: `new stop ${P.stop.toFixed(2)} (${P.trailDeltaBips} bips)` });
      }
    }

    saveState(state);
    return { ok: true };
  }
};
