import axios from 'axios';
import * as crypto from 'crypto';
import { CFG } from './config.js';

const api = axios.create({ baseURL: CFG.BINANCE.BASE_URL, timeout: 10000 });

api.interceptors.request.use((cfg) => {
  if (CFG.BINANCE.KEY) cfg.headers['X-MBX-APIKEY'] = CFG.BINANCE.KEY;
  return cfg;
});

function sign(qs) {
  return crypto.createHmac('sha256', CFG.BINANCE.SECRET).update(qs).digest('hex');
}

function toQS(params) { return new URLSearchParams(params).toString(); }

export const binance = {
  async serverTime() {
    const { data } = await api.get('/api/v3/time');
    return data.serverTime;
  },
  async accountInfo() {
    const ts = Date.now();
    const q = toQS({ timestamp: ts });
    const { data } = await api.get(`/api/v3/account?${q}&signature=${sign(q)}`);
    return data;
  },
  async placeOrder(o) {
    const ts = Date.now();
    const params = { ...o, timestamp: ts, newOrderRespType: 'RESULT' };
    const q = toQS(params);
    const { data } = await api.post(`/api/v3/order`, q + `&signature=${sign(q)}`);
    return data;
  },
  async cancelReplace(o) {
    const ts = Date.now();
    const params = { ...o, timestamp: ts, newOrderRespType: 'RESULT' };
    const q = toQS(params);
    const { data } = await api.post(`/api/v3/order/cancelReplace`, q + `&signature=${sign(q)}`);
    return data;
  },
  async cancelOpenOrders(symbol) {
    const ts = Date.now();
    const q = toQS({ symbol, timestamp: ts });
    const { data } = await api.delete(`/api/v3/openOrders?${q}&signature=${sign(q)}`);
    return data;
  }
};
